/*
 * Decompiled with CFR 0.152.
 */
package ninja.leaping.configurate.objectmapping.serialize;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.reflect.TypeToken;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import ninja.leaping.configurate.ConfigurationNode;
import ninja.leaping.configurate.SimpleConfigurationNode;
import ninja.leaping.configurate.Types;
import ninja.leaping.configurate.objectmapping.InvalidTypeException;
import ninja.leaping.configurate.objectmapping.ObjectMappingException;
import ninja.leaping.configurate.objectmapping.serialize.ConfigSerializable;
import ninja.leaping.configurate.objectmapping.serialize.TypeSerializer;
import ninja.leaping.configurate.objectmapping.serialize.TypeSerializerCollection;
import ninja.leaping.configurate.util.EnumLookup;

public class TypeSerializers {
    private static final TypeSerializerCollection DEFAULT_SERIALIZERS = new TypeSerializerCollection(null);

    public static TypeSerializerCollection getDefaultSerializers() {
        return DEFAULT_SERIALIZERS;
    }

    static {
        DEFAULT_SERIALIZERS.registerType(TypeToken.of(URI.class), new URISerializer());
        DEFAULT_SERIALIZERS.registerType(TypeToken.of(URL.class), new URLSerializer());
        DEFAULT_SERIALIZERS.registerType(TypeToken.of(UUID.class), new UUIDSerializer());
        DEFAULT_SERIALIZERS.registerPredicate(input -> input.getRawType().isAnnotationPresent(ConfigSerializable.class), new AnnotatedObjectSerializer());
        DEFAULT_SERIALIZERS.registerType(TypeToken.of(Number.class), new NumberSerializer());
        DEFAULT_SERIALIZERS.registerType(TypeToken.of(String.class), new StringSerializer());
        DEFAULT_SERIALIZERS.registerType(TypeToken.of(Boolean.class), new BooleanSerializer());
        DEFAULT_SERIALIZERS.registerType(new TypeToken<Map<?, ?>>(){}, new MapSerializer());
        DEFAULT_SERIALIZERS.registerType(new TypeToken<List<?>>(){}, new ListSerializer());
        DEFAULT_SERIALIZERS.registerType(new TypeToken<Enum<?>>(){}, new EnumValueSerializer());
        DEFAULT_SERIALIZERS.registerType(TypeToken.of(Pattern.class), new PatternSerializer());
    }

    private static class PatternSerializer
    implements TypeSerializer<Pattern> {
        private PatternSerializer() {
        }

        @Override
        public Pattern deserialize(TypeToken<?> type, ConfigurationNode value) throws ObjectMappingException {
            try {
                return Pattern.compile(value.getString());
            }
            catch (PatternSyntaxException ex) {
                throw new ObjectMappingException(ex);
            }
        }

        @Override
        public void serialize(TypeToken<?> type, Pattern obj, ConfigurationNode value) throws ObjectMappingException {
            value.setValue(obj.pattern());
        }
    }

    private static class UUIDSerializer
    implements TypeSerializer<UUID> {
        private UUIDSerializer() {
        }

        @Override
        public UUID deserialize(TypeToken<?> type, ConfigurationNode value) throws ObjectMappingException {
            try {
                return UUID.fromString(value.getString());
            }
            catch (IllegalArgumentException ex) {
                throw new ObjectMappingException("Value not a UUID", ex);
            }
        }

        @Override
        public void serialize(TypeToken<?> type, UUID obj, ConfigurationNode value) throws ObjectMappingException {
            value.setValue(obj.toString());
        }
    }

    private static class URLSerializer
    implements TypeSerializer<URL> {
        private URLSerializer() {
        }

        @Override
        public URL deserialize(TypeToken<?> type, ConfigurationNode value) throws ObjectMappingException {
            URL url;
            String plainUrl = value.getString();
            if (plainUrl == null) {
                throw new ObjectMappingException("No value present in node " + value);
            }
            try {
                url = new URL(plainUrl);
            }
            catch (MalformedURLException e) {
                throw new ObjectMappingException("Invalid URL string provided for " + value.getKey() + ": got " + plainUrl);
            }
            return url;
        }

        @Override
        public void serialize(TypeToken<?> type, URL obj, ConfigurationNode value) throws ObjectMappingException {
            value.setValue(obj.toString());
        }
    }

    private static class URISerializer
    implements TypeSerializer<URI> {
        private URISerializer() {
        }

        @Override
        public URI deserialize(TypeToken<?> type, ConfigurationNode value) throws ObjectMappingException {
            URI uri;
            String plainUri = value.getString();
            if (plainUri == null) {
                throw new ObjectMappingException("No value present in node " + value);
            }
            try {
                uri = new URI(plainUri);
            }
            catch (URISyntaxException e) {
                throw new ObjectMappingException("Invalid URI string provided for " + value.getKey() + ": got " + plainUri);
            }
            return uri;
        }

        @Override
        public void serialize(TypeToken<?> type, URI obj, ConfigurationNode value) throws ObjectMappingException {
            value.setValue(obj.toString());
        }
    }

    private static class AnnotatedObjectSerializer
    implements TypeSerializer<Object> {
        private AnnotatedObjectSerializer() {
        }

        @Override
        public Object deserialize(TypeToken<?> type, ConfigurationNode value) throws ObjectMappingException {
            Class<?> clazz = this.getInstantiableType(type, value.getNode("__class__").getString());
            return value.getOptions().getObjectMapperFactory().getMapper(clazz).bindToNew().populate(value);
        }

        private Class<?> getInstantiableType(TypeToken<?> type, String configuredName) throws ObjectMappingException {
            Class retClass;
            if (type.getRawType().isInterface() || Modifier.isAbstract(type.getRawType().getModifiers())) {
                if (configuredName == null) {
                    throw new ObjectMappingException("No available configured type for instances of " + type);
                }
                try {
                    retClass = Class.forName(configuredName);
                }
                catch (ClassNotFoundException e) {
                    throw new ObjectMappingException("Unknown class of object " + configuredName, e);
                }
            } else {
                retClass = type.getRawType();
            }
            return retClass;
        }

        @Override
        public void serialize(TypeToken<?> type, Object obj, ConfigurationNode value) throws ObjectMappingException {
            if (type.getRawType().isInterface() || Modifier.isAbstract(type.getRawType().getModifiers())) {
                value.getNode("__class__").setValue(type.getRawType().getCanonicalName());
            }
            value.getOptions().getObjectMapperFactory().getMapper(obj.getClass()).bind(obj).serialize(value);
        }
    }

    private static class ListSerializer
    implements TypeSerializer<List<?>> {
        private ListSerializer() {
        }

        @Override
        public List<?> deserialize(TypeToken<?> type, ConfigurationNode value) throws ObjectMappingException {
            if (!(type.getType() instanceof ParameterizedType)) {
                throw new ObjectMappingException("Raw types are not supported for collections");
            }
            TypeToken entryType = type.resolveType(List.class.getTypeParameters()[0]);
            TypeSerializer entrySerial = value.getOptions().getSerializers().get(entryType);
            if (entrySerial == null) {
                throw new ObjectMappingException("No applicable type serializer for type " + entryType);
            }
            if (value.hasListChildren()) {
                List<? extends ConfigurationNode> values = value.getChildrenList();
                ArrayList ret = new ArrayList(values.size());
                for (ConfigurationNode configurationNode : values) {
                    ret.add(entrySerial.deserialize(entryType, configurationNode));
                }
                return ret;
            }
            Object unwrappedVal = value.getValue();
            if (unwrappedVal != null) {
                return Lists.newArrayList((Object[])new Object[]{entrySerial.deserialize(entryType, value)});
            }
            return new ArrayList();
        }

        @Override
        public void serialize(TypeToken<?> type, List<?> obj, ConfigurationNode value) throws ObjectMappingException {
            if (!(type.getType() instanceof ParameterizedType)) {
                throw new ObjectMappingException("Raw types are not supported for collections");
            }
            TypeToken entryType = type.resolveType(List.class.getTypeParameters()[0]);
            TypeSerializer<?> entrySerial = value.getOptions().getSerializers().get(entryType);
            if (entrySerial == null) {
                throw new ObjectMappingException("No applicable type serializer for type " + entryType);
            }
            value.setValue(ImmutableList.of());
            for (Object ent : obj) {
                entrySerial.serialize(entryType, ent, value.getAppendedNode());
            }
        }
    }

    private static class MapSerializer
    implements TypeSerializer<Map<?, ?>> {
        private MapSerializer() {
        }

        @Override
        public Map<?, ?> deserialize(TypeToken<?> type, ConfigurationNode node) throws ObjectMappingException {
            LinkedHashMap ret = new LinkedHashMap();
            if (node.hasMapChildren()) {
                if (!(type.getType() instanceof ParameterizedType)) {
                    throw new ObjectMappingException("Raw types are not supported for collections");
                }
                TypeToken key = type.resolveType(Map.class.getTypeParameters()[0]);
                TypeToken value = type.resolveType(Map.class.getTypeParameters()[1]);
                TypeSerializer keySerial = node.getOptions().getSerializers().get(key);
                TypeSerializer valueSerial = node.getOptions().getSerializers().get(value);
                if (keySerial == null) {
                    throw new ObjectMappingException("No type serializer available for type " + key);
                }
                if (valueSerial == null) {
                    throw new ObjectMappingException("No type serializer available for type " + value);
                }
                for (Map.Entry<Object, ? extends ConfigurationNode> ent : node.getChildrenMap().entrySet()) {
                    Object keyValue = keySerial.deserialize(key, SimpleConfigurationNode.root().setValue(ent.getKey()));
                    Object valueValue = valueSerial.deserialize(value, ent.getValue());
                    if (keyValue == null || valueValue == null) continue;
                    ret.put(keyValue, valueValue);
                }
            }
            return ret;
        }

        @Override
        public void serialize(TypeToken<?> type, Map<?, ?> obj, ConfigurationNode node) throws ObjectMappingException {
            if (!(type.getType() instanceof ParameterizedType)) {
                throw new ObjectMappingException("Raw types are not supported for collections");
            }
            TypeToken key = type.resolveType(Map.class.getTypeParameters()[0]);
            TypeToken value = type.resolveType(Map.class.getTypeParameters()[1]);
            TypeSerializer<?> keySerial = node.getOptions().getSerializers().get(key);
            TypeSerializer<?> valueSerial = node.getOptions().getSerializers().get(value);
            if (keySerial == null) {
                throw new ObjectMappingException("No type serializer available for type " + key);
            }
            if (valueSerial == null) {
                throw new ObjectMappingException("No type serializer available for type " + value);
            }
            node.setValue(ImmutableMap.of());
            for (Map.Entry<?, ?> ent : obj.entrySet()) {
                SimpleConfigurationNode keyNode = SimpleConfigurationNode.root();
                keySerial.serialize(key, ent.getKey(), keyNode);
                valueSerial.serialize(value, ent.getValue(), node.getNode(keyNode.getValue()));
            }
        }
    }

    private static class EnumValueSerializer
    implements TypeSerializer<Enum> {
        private EnumValueSerializer() {
        }

        @Override
        public Enum deserialize(TypeToken<?> type, ConfigurationNode value) throws ObjectMappingException {
            String enumConstant = value.getString();
            if (enumConstant == null) {
                throw new ObjectMappingException("No value present in node " + value);
            }
            Optional<Enum> ret = EnumLookup.lookupEnum(type.getRawType().asSubclass(Enum.class), enumConstant);
            if (!ret.isPresent()) {
                throw new ObjectMappingException("Invalid enum constant provided for " + value.getKey() + ": Expected a value of enum " + type + ", got " + enumConstant);
            }
            return ret.get();
        }

        @Override
        public void serialize(TypeToken<?> type, Enum obj, ConfigurationNode value) throws ObjectMappingException {
            value.setValue(obj.name());
        }
    }

    private static class BooleanSerializer
    implements TypeSerializer<Boolean> {
        private BooleanSerializer() {
        }

        @Override
        public Boolean deserialize(TypeToken<?> type, ConfigurationNode value) throws InvalidTypeException {
            return value.getBoolean();
        }

        @Override
        public void serialize(TypeToken<?> type, Boolean obj, ConfigurationNode value) {
            value.setValue(Types.asBoolean(obj));
        }
    }

    private static class NumberSerializer
    implements TypeSerializer<Number> {
        private NumberSerializer() {
        }

        @Override
        public Number deserialize(TypeToken<?> type, ConfigurationNode value) throws InvalidTypeException {
            Class clazz = (type = type.wrap()).getRawType();
            if (Integer.class.equals((Object)clazz)) {
                return value.getInt();
            }
            if (Long.class.equals((Object)clazz)) {
                return value.getLong();
            }
            if (Short.class.equals((Object)clazz)) {
                return (short)value.getInt();
            }
            if (Byte.class.equals((Object)clazz)) {
                return (byte)value.getInt();
            }
            if (Float.class.equals((Object)clazz)) {
                return Float.valueOf(value.getFloat());
            }
            if (Double.class.equals((Object)clazz)) {
                return value.getDouble();
            }
            return null;
        }

        @Override
        public void serialize(TypeToken<?> type, Number obj, ConfigurationNode value) {
            value.setValue(obj);
        }
    }

    private static class StringSerializer
    implements TypeSerializer<String> {
        private StringSerializer() {
        }

        @Override
        public String deserialize(TypeToken<?> type, ConfigurationNode value) throws InvalidTypeException {
            return value.getString();
        }

        @Override
        public void serialize(TypeToken<?> type, String obj, ConfigurationNode value) {
            value.setValue(obj);
        }
    }
}

